<?php

namespace Flow\FlowPayment\Controller\Payment;

class FlowReturn extends \Magento\Framework\App\Action\Action
{
    protected $orderRepository;
    protected $invoiceService;
    protected $dbTransaction;
    protected $messageManager;
    protected $helper;
    protected $logger;
    protected $quoteRepository;
    protected $eventManager;
    protected $customer;

    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Checkout\Model\Cart $cart,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Controller\Result\JsonFactory $resultJsonFactory,
        \Magento\Quote\Model\QuoteManagement $quoteManagement,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Flow\FlowPayment\Model\Config\ConfigProvider $configProvider,
        \Magento\Sales\Api\OrderRepositoryInterface $orderRepository,
        \Magento\Sales\Model\Service\InvoiceService $invoiceService,
        \Magento\Framework\DB\Transaction $transaction,
        \Flow\FlowPayment\Helper\Helper $helper,
        \Psr\Log\LoggerInterface $logger,
        \Magento\Quote\Api\CartRepositoryInterface $quoteRepository,
        \Magento\Framework\Event\ManagerInterface $eventManager,
        \Magento\Customer\Model\Session $customerSession
    )
    {
        parent::__construct($context);

        $this->cart = $cart;
        $this->checkoutSession = $checkoutSession;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->quoteManagement = $quoteManagement;
        $this->storeManager = $storeManager;
        $this->configProvider = $configProvider;
        $this->orderRepository = $orderRepository;
        $this->invoiceService = $invoiceService;
        $this->dbTransaction = $transaction;
        $this->messageManager = $context->getMessageManager();
        $this->helper = $helper;
        $this->logger = $logger;
        $this->quoteRepository = $quoteRepository;
        $this->eventManager = $eventManager;
        $this->customer = $customerSession;
    }

    public function execute()
    {
        try {

            if (!isset($_POST["token"])) {
                throw new \Exception("No token");
            }
    
            $token = $_POST["token"];
    
            $this->logger->debug('Entering the return controller...', ["token" => $token]);
    
            $config = $this->configProvider->getPluginConfig();
            $successfulOrderState = $config['SUCCESSFUL_STATE'];
            $failedOrderState = $config['FAILED_STATE'];
            $flowApi = $this->configProvider->getFlowApi();
    
            $this->logger->debug("Calling Flow Service order/token with params... ".json_encode(["token" => $token]));
            $response = $flowApi->getOrderStatus($token);
            $this->logger->debug("Flow Response: ".json_encode($response));
            $statusInFlow = $response["status"];
    
            $order = $this->orderRepository->get($response["commerce_order"]);
            $this->logger->debug('Order found. Id='.$order->getId());

            $orderCurrency = $order->getOrderCurrencyCode();
            $orderAmount = $this->helper->getOrderAmount($order->getGrandTotal(), $orderCurrency);

            if ($this->helper->isPaidInFlow($statusInFlow)) {
                $this->logger->debug('Order is paid in flow.');

                if ($orderAmount !== (float) $response["amount"]) {
                    throw new \Exception("The order total and the total paid differ. Aborting...");
                }
                
                
                //If the order is paid in flow and processing in the store, means everything is ready to go, so we 
                //immediately redirect to the success page.
                if ($order->getStatus() == $successfulOrderState) {
                    $this->logger->debug('Order has the correct status.');
                    $this->deleteCart();

                    return $this->redirectToSuccess($order);
                //if for some reason the confirmation page wasn't reached, we accept the order here.
                } else if ($order->getStatus() == \Magento\Sales\Model\Order::STATE_PENDING_PAYMENT) {
                    $this->logger->debug('Confirmation page was not reached. Order has pending status still');
                    $this->acceptOrder($order, $successfulOrderState);
                    return $this->redirectToSuccess($order);
                }
    
            } else if ($this->helper->isPendingInFlow($statusInFlow)) {
                if ($this->helper->userGeneratedCoupon($response)) {
                    $this->deleteCart();
                    $this->logger->debug('User generated coupon...');

                    if (!empty($config["CUSTOM_RETURN_URL"])) {
                        $this->logger->debug('User has a custom return url. Redirecting there...');
                        return $this->_redirect($config["CUSTOM_RETURN_URL"]);
                    }

                    $this->redirectToCoupon();

                } else if($this->helper->userCanceledOrder($response)) {
                    $this->logger->debug("User canceled order");
                    $this->restoreCart($order);
                    return $this->redirectToCheckout();
                } else {
                    return $this->redirectToError();
                }
    
            } else if ($this->helper->isRejectedInFlow($statusInFlow)) {
    
                //If the order doesn't have already the canceled status, we cancel it here.
                if ($order->getStatus() != $failedOrderState) {
                    $this->rejectOrder($order, $failedOrderState);
                }
                
                $this->restoreCart($order);
                return $this->redirectToFailure("Ha ocurrido un error procesando su pago.");
            }

        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            return $this->redirectToError();
        }

    }

    private function rejectOrder($order, $state)
    {
        $this->logger->debug('Rejecting order...');
        $order->setState($state)
        ->setStatus($state)
        ->addStatusToHistory($state,  "Order rejected in Flow");
        $order->save();
        $this->restoreCart($order);
    }

    private function acceptOrder($order, $state)
    {
        $this->logger->debug('Accepting order...');

        $order->setState($state)
        ->setStatus($state)
        ->addStatusToHistory("processing",  "Order paid successfully in Flow");

        $payment = $order->getPayment();
        $payment->setLastTransId($order->getId());
        $payment->setTransactionId($order->getId());
        $payment->addTransaction(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_CAPTURE, null, true);

        $order->save();

        $invoice = $this->invoiceService->prepareInvoice($order);
        $invoice->setTransactionId($order->getId());
        $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_OFFLINE);
        $invoice->register();

        $savedTransaction = $this->dbTransaction->addObject($invoice)->addObject($invoice->getOrder());
        $savedTransaction->save();

        $this->deleteCart();
    }

    private function deleteCart()
    {
        $this->logger->debug('Deleting cart...');
        $this->checkoutSession->getQuote()->setIsActive(false)->save();
    }

    private function redirectToFailure($message)
    {
        $this->logger->debug('Redirecting to failure...');
        $this->messageManager->addError($message);
        $this->_redirect('checkout/onepage/failure', ['_secure' => false]);
    }

    private function redirectToSuccess($order)
    {
        $this->logger->debug('Prepare data redirect to checkout/onepage/success order '.$order->getId().'...');
        $quote = $this->cart->getQuote();
        $orderCurrency = $order->getOrderCurrencyCode();
        $orderAmount = $this->helper->getOrderAmount($order->getGrandTotal(), $orderCurrency);

        $this->checkoutSession->setLastQuoteId($quote->getId());
        $this->checkoutSession->setLastSuccessQuoteId($quote->getId());
        $this->checkoutSession->setLastOrderId($order->getId());
        //$session->setLastOrderId($orderId);
        $this->checkoutSession->setLastRealOrderId($order->getIncrementId());
        //$session->setLastRealOrderId($incrementId);
        $this->checkoutSession->setLastOrderStatus($order->getStatus());
        $this->checkoutSession->setGrandTotal($orderAmount);
        $this->restoreCustomerSession($order);
        $this->logger->debug('Redirecting to success...');
        $this->_redirect('checkout/onepage/success', ['_secure' => false]);
    }

    private function redirectToCoupon()
    {
        $this->logger->debug('Redirecting to coupon...');
        $this->_redirect('flowpayment/coupon/index', ['_secure' => false]);
    }

    private function redirectToCheckout()
    {
        $this->logger->debug('Redirecting to checkout...');
        $this->_redirect("checkout");
    }

    private function setUpProductionEnvSimulation(&$flowData){
        $this->logger->debug('Setting up simulation...');
        $flowData['pending_info']['media'] = $flowData['payment']['media'];
        $flowData['payment']['media'] = '';
        $flowData['status'] = 1;
    }

    private function redirectToError()
    {
        $this->_redirect('flowpayment/error/index', ['_secure' => false]);
    }

    private function restoreCart($order)
    {
        $this->logger->debug('Prepare data...'.$order->getId());
        if ($order->getId()) {
            $quote = $this->cart->getQuote();
            $this->logger->debug('Entering restore quote...'.$order->getQuoteId());
            $quote = $this->quoteRepository->get($order->getQuoteId());
            $quote->setIsActive(1)->setReservedOrderId(null);
            $this->quoteRepository->save($quote);
            $this->checkoutSession->replaceQuote($quote)->unsLastRealOrderId();
            $this->eventManager->dispatch('restore_quote', ['order' => $order, 'quote' => $quote]);
            $this->logger->debug('Finalize restore quote...'.json_encode($quote));
            $this->restoreCustomerSession($order);
            return true;
        }
        $this->logger->debug('Finalize restore quote with false...');
        return false;
    }

    private function restoreCustomerSession($order)
    {
		//Restoring the customer so magento doesn't log out
        $this->logger->debug('Entry restore session user...');
		if (!$this->customer->getId()) {
			$customerId = $order->getCustomerId();
            $this->logger->debug('Customer id: '.$customerId);
			$this->customer->setId($customerId);
            $this->logger->debug('End set customer id...');
		}
        $this->logger->debug('Finalize restore session customer...');
    }
}
