<?php

namespace Flow\FlowPayment\Controller\Payment;

class Confirm extends \Magento\Framework\App\Action\Action
{
    protected $orderRepository;
    protected $invoiceService;
    protected $dbTransaction;
    protected $messageManager;
    protected $helper;  
    protected $logger;

    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Checkout\Model\Cart $cart,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Controller\Result\JsonFactory $resultJsonFactory,
        \Magento\Quote\Model\QuoteManagement $quoteManagement,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Flow\FlowPayment\Model\Config\ConfigProvider $configProvider,
        \Magento\Sales\Api\OrderRepositoryInterface $orderRepository,
        \Magento\Sales\Model\Service\InvoiceService $invoiceService,
        \Magento\Framework\DB\Transaction $transaction,
        \Flow\FlowPayment\Helper\Helper $helper,
        \Psr\Log\LoggerInterface $logger
    )
    {
        parent::__construct($context);

        $this->cart = $cart;
        $this->checkoutSession = $checkoutSession;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->quoteManagement = $quoteManagement;
        $this->storeManager = $storeManager;
        $this->configProvider = $configProvider;
        $this->orderRepository = $orderRepository;
        $this->invoiceService = $invoiceService;
        $this->dbTransaction = $transaction;
        $this->messageManager = $context->getMessageManager();
        $this->helper = $helper;
        $this->logger = $logger;
    }

    public function execute()
    {
        try {

            if (!isset($_POST["token"])) {
                throw new \Exception("No token");
            }

            $token = $_POST["token"];

            $this->logger->debug('Entering the confirm controller...', ['token' => $token]);

            $config = $this->configProvider->getPluginConfig();
            $successfulOrderState = $config['SUCCESSFUL_STATE'];
            $failedOrderState = $config['FAILED_STATE'];
            $flowApi = $this->configProvider->getFlowApi();

            $this->logger->debug("Calling Flow Service order/token with params... ".json_encode(["token" => $token]));
            $response = $flowApi->getOrderStatus($token);
            $this->logger->debug("Flow Response: ".json_encode($response));

            $statusInFlow = $response["status"];

            $order = $this->orderRepository->get($response["commerce_order"]);
            $this->logger->debug('Order found. id='.$order->getId());

            $orderCurrency = $order->getOrderCurrencyCode();
            $orderAmount = $this->helper->getOrderAmount($order->getGrandTotal(), $orderCurrency);

            if ($this->helper->isPaidInFlow($statusInFlow)) {
                $this->logger->debug('Order is paid in Flow.');

                if ($orderAmount !== (float) $response["amount"]) {
                    throw new \Exception("The order total and the total paid differ. Aborting...");
                }

                $this->logger->debug('Accepting order...');

                $order->setState($successfulOrderState)
                ->setStatus($successfulOrderState)
                ->addStatusToHistory($successfulOrderState,  "Order paid successfully in Flow");

                $payment = $order->getPayment();
                $payment->setLastTransId($response["id"]);
                $payment->setTransactionId($response["id"]);
                $payment->addTransaction(\Magento\Sales\Model\Order\Payment\Transaction::TYPE_CAPTURE, null, true);

                $order->save();

                $invoice = $this->invoiceService->prepareInvoice($order);
                $invoice->setTransactionId($response["id"]);
                $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_OFFLINE);
                $invoice->register();

                $savedTransaction = $this->dbTransaction->addObject($invoice)->addObject($invoice->getOrder());
                $savedTransaction->save();

            } else if ($this->helper->isRejectedInFlow($statusInFlow)) {

                $this->logger->debug('Order is rejected in Flow.');
                $this->logger->debug('Rejecting order...');
                $order->setState($failedOrderState)
                ->setStatus($failedOrderState)
                ->addStatusToHistory($failedOrderState,  "Order rejected in Flow");
                $order->save();
            } else if ($this->helper->isPendingInFlow($statusInFlow)) {
                $this->logger->debug('Order is pending in Flow. Doing nothing...');
            }

        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

    }
}
