<?php

namespace Flow\FlowPayment\Model\Config;

class ConfigProvider implements \Magento\Checkout\Model\ConfigProviderInterface {

    protected $_assetsRepo;

    public function __construct(
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfigInterface
    ) 
    {
        $this->scopeConfigInterface = $scopeConfigInterface;
    }

    public function getConfig()
    {
        $conf = 'payment/flow_flowpayment/';
        $om = \Magento\Framework\App\ObjectManager::getInstance();
        $fileSystem = $om->create('\Magento\Framework\Filesystem');
        $mediaPath = $fileSystem->getDirectoryRead(\Magento\Framework\App\Filesystem\DirectoryList::MEDIA)->getAbsolutePath();

        $storeManager = $om->get('Magento\Store\Model\StoreManagerInterface'); 
        $currentStore = $storeManager->getStore();

        $request = $om->get('Magento\Framework\App\RequestInterface');
        $this->_assetsRepo = $om->get('Magento\Framework\View\Asset\Repository');
        $params = array();
        $params = array_merge(array('_secure' => $request->isSecure()), $params);

        // We first check if the user uploaded a custom logo
        if (file_exists($mediaPath."flowpayment/customlogo/default/custom-logo.png")) {
            $logoUrl = $currentStore->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA)."flowpayment/customlogo/default/custom-logo.png";
        } else {
            $logoUrl = $this->_assetsRepo->getUrlWithParams('Flow_FlowPayment::images/logo.png', $params);
        }

        return array(
            'pluginConfigFlowPayment' => array(
                'createPayment' => 'flowpayment/payment/initpayment',
                'paymentName' => $this->scopeConfigInterface->getValue($conf.'payment_name'),
                'logo' => $logoUrl
            )
        );
    }

    public function getPluginConfig()
    {
        $conf = 'payment/flow_flowpayment/';
        return array(
			'ENVIRONMENT' => $this->scopeConfigInterface->getValue($conf.'environment'),
            'API_KEY' => $this->scopeConfigInterface->getValue($conf.'api_key'),
            'RETURN_URL' => 'flowpayment/payment/flowreturnpayment',
            'CONFIRM_URL' => 'flowpayment/payment/confirmpayment',
            'CUSTOM_RETURN_URL' => $this->scopeConfigInterface->getValue($conf.'return_url'),
            'SUCCESSFUL_STATE' => $this->scopeConfigInterface->getValue($conf.'successful_state'),
            'FAILED_STATE' => $this->scopeConfigInterface->getValue($conf.'failed_state')
        );
    }

    public function getFlowApi()
    {
        $config = $this->getPluginConfig();
        $endpoint = $config["ENVIRONMENT"] === "TEST" ? "https://sandbox.flow.cl/api/v2" : "https://www.flow.cl/api/v2";
        $flowApi = new FlowApi($endpoint, $config["API_KEY"]);
        return $flowApi;
    }
}

class FlowApi
{
    private $endpoint;
    private $apiKey;
    private $secretKey;

    public function __construct($endpoint, $apiKey, $secretKey = "")
    {
        $this->endpoint = $endpoint;
        $this->apiKey = $apiKey;
        $this->secretKey = $secretKey;
    }

    public function order($data)
    {
        $endpoint = $this->getEndpoint("order/create");
        return $this->processResponse($this->httpRequest($endpoint, "POST", $data));
    }

    public function getOrderStatus($token)
    {
        $endpoint = $this->getEndpoint("order/token/".$token);
        return $this->processResponse($this->httpRequest($endpoint, "GET"));
    }

    private function getHeaders()
    {
        return array(
            "Authorization: Basic ".$this->getAuthHeader(),
            "Content-type: application/json"
        );
    }

    private function getAuthHeader()
    {
        return base64_encode($this->apiKey.":".$this->secretKey);
    }
    
    private function getEndpoint($service)
    {
        return $this->endpoint."/".$service;
    }

    private function processResponse($response)
    {
        //if (!empty($response["error"])) {
            //error_log("Error communicating with Flow: ".$response["error"]);
          //  throw new \Exception("Error communicating with Flow");
        //}

        $responseDecoded = json_decode($response["output"], TRUE);
        $info = $response["info"];
        $http_code = $info["http_code"];
        if($response["error"] !== null){
            $http_code = 443;
        }

        if ((!empty($responseDecoded["code"]) && !empty($responseDecoded["message"])) || empty($responseDecoded)) {
            error_log("Error communicating with Flow: ".$response["output"]);
            throw new \Exception(base64_encode($this->getFlowErrorMessage($responseDecoded,$http_code)), 1000);
        }

        return $responseDecoded;
    }

    private function getFlowErrorMessage($flowError, $http_code)
    {
        $message = null;
        switch ($http_code)
        {
            case 401:
            {
                $message = "El API-Key es inv&aacute;lido, verifique sus credenciales en la configuraci&oacute;n de su cuenta de Flow. Para m&aacute;s informaci&oacute;n visite <a href='https://ayuda.flow.cl/'>https://ayuda.flow.cl/</a> y revise la secci&oacute;n de ayuda t&eacute;cnica.";
                break;
            }
            case 404:
            {
                $message = "El servicio no est&aacute; disponible. Para m&aacute;s informaci&oacute;n comun&iacute;quese con <a href='mailto:soporte@flow.cl'>soporte@flow.cl</a>";
                break;
            }
            case 443;
            {
                $message = "Ha ocurrido un error de comunicación con Flow. Para más información visite <a href='https://ayuda.flow.cl/'>https://ayuda.flow.cl/</a> y revise la sección de ayuda t&eacute;cnica." ;
                break;
            }
        }

        if ($flowError["code"] === 104 && !empty($flowError["errors"])) {
            $businessErrors = $flowError["errors"][0];
            $subMessage = $businessErrors["message"];
            switch ($businessErrors["field"]) {
                case "currency":
                case "amount": {
                    $visite = "Para m&aacute;s informaci&oacute;n visite <a href='https://ayuda.flow.cl/'>https://ayuda.flow.cl/</a> y revise la secci&oacute;n de ayuda t&eacute;cnica.";
                    break;
                }
                default: {
                    $visite = "Para m&aacute;s informaci&oacute;n comun&iacute;quese con <a href='mailto:soporte@flow.cl'>soporte@flow.cl</a>";
                    break;
                }
            }
            $message = "Ha ocurrido un error in&eacute;sperado: $subMessage.<br>$visite";
        }

        return $message;
    }

	/**
	 * Funcion que hace el llamado via http POST
	 * @param string $url url a invocar
	 * @param array $params los datos a enviar
	 * @return array el resultado de la llamada
	 * @throws Exception
	 */
    private function httpRequest($url, $type, $params = array())
    {
        $ch = curl_init();

        if ($type === "POST") {
            $params = json_encode($params);
            curl_setopt($ch, CURLOPT_POST, TRUE);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        }

		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $this->getHeaders());
        $output = curl_exec($ch);
        
        $error = null;
		$info = curl_getinfo($ch);
        
        if ($output === FALSE) {
            $error = curl_error($ch);
        }

        curl_close($ch);

		return array(
            "output" => $output, 
            "info" => $info,
            "error" => $error
        );
	}
}